///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   XSVD.C - Test Extraction Ephemeris Processing
// 
//
//   DESCRIPTION
//
//   Functions to extract, decode, and write ephemeris data to a file.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\XSVD.C_V  $
//   
//      Rev 1.15   09 Nov 1998 10:48:12   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.5   Jul 09 1997 10:58:52   COLEJ
//    
//   
//      Rev 1.4   Feb 12 1997 16:06:00   COLEJ
//    
//   
//      Rev 1.3   Aug 19 1996 17:12:34   COLEJ
//    
//   
//      Rev 1.2   Aug 16 1996 16:57:14   COLEJ
//   Added extraction of raw almanac to a file.
//   
//   
//      Rev 1.0   Aug 14 1996 18:20:16   COLEJ
//   New SV data extraction routines ( msg1102)
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>                     
#include <string.h>
#include <math.h>
#include <graph.h>

#include "xsvd.h"
#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"   
#include "util.h"


unsigned short     sfid=0,   // global value for sub frame id - for multiple function calls
                   svid=0,   // global value for almanac svid number (prn)
                   pa;       // preamble of current message
                   
unsigned long      ephavail=0,             // availability 32bit flags for epemeris sv
                   almavail=0;             // availability 32bit flags for almanac sv
                   
tBOOL              COLLECT_EPH = FALSE,    // flag for need to collect epheneris data
                   COLLECT_ALM = FALSE,    // flag for need to collect almanac data
                   COLLECT_UTC = FALSE;    // flag for need to collect utc data
                   
tBOOL              EPHFILE_INI = FALSE,    // flag for tmp files initialized
                   ALMFILE_INI = FALSE;



struct {
   tBOOL        utcmem;              // out of memory flag for utc structure
   utctype      *utcptr;             // pointer to utc structure.
}          utcstructptr;


unsigned long   current1,          // current data word 1
                current2;          // current data word 2

extern FILE   *stream3, *stream6, *stream7, *stream8, *almdata;
extern void   clear_message_line  (void);


/////////////////////////////////////////////////////////////////////////////////////////////////
//   
//   Extract SV Data and Output to File
//      
void XSVD(tMSG1102 *Msg)
{
   unsigned short  ch=0,      // current channel index
                   w=0,       // current dataword index (0 or 1)
                   index=0,   // current index to switch
                   dwvalid=0, // validity test for data word
                   sfn=0,     // current sub frame number
                   prn=0;     // 1102 PRN feild
   
   unsigned long   dw[2],             // array for data words
                   epharray[26],      // array for a local copy of ephemeris data initialization
                   almarray[11],      // array for a local copy of almanac data initialization
                   rawalmarray[11];   // array for a local copy of raw almanac data words
   long os=0;                         // offset from beginning of file 
   int i,count = 0;

   
   // initialize tmp files if necessary
   if( COLLECT_EPH && (EPHFILE_INI == FALSE) ){
      EPHFILE_INI = TRUE;
      rewind(stream6);
      for(i=0; i<26; i++)                         // for each peice of data at offset
         epharray[i] = 0;                         // set to 0
      count = fwrite( epharray, 26 * sizeof (unsigned long),32, stream6 );   // write for each sv
      if(_Debug) showmessage( "Number items wrote to (eph.tmp):", 2, count );
   }  // end initialization for eph.tmp

   if( COLLECT_ALM && (ALMFILE_INI == FALSE) ){
      ALMFILE_INI = TRUE;
      rewind(stream7);
      rewind(stream8);                            // raw almanac
      for(i=0; i<11; i++)                         // for each peice of data at offset
         almarray[i] = 0;                         // set to 0
         rawalmarray[i] = 0;
      count = fwrite( almarray, 11 * sizeof (unsigned long),32, stream7 );      // write for each sv
      if(_Debug) showmessage( "Number items wrote to (alm.tmp):", 2, count );
      count = fwrite( rawalmarray, 11 * sizeof (unsigned long),32, stream8 );   // write for each sv
      if(_Debug) showmessage( "Number items wrote to (almanac.dat):", 2, count );
   }  // end initialization for alm.tmp

   
   // setup and extract data for each channel
   for(ch=0; ch<GI_NUM_HW_CHAN; ch++){
     prn = Msg->ChannelData[ch].PRN -1; // offset by -1 for index to array
     sfn = Msg->ChannelData[ch].FrameIndex;
     for (w=0; w<2; w++){
       dw[w] = Msg->ChannelData[ch].DataWord[w];
       dwvalid = (unsigned short) gbits(dw[w],-8,-7);
       if ((dw[0] == 0) || (dwvalid != 0x01)) break;
       dw[w] &= 0x00ffffff;
       index = ((w==1) && (dw[w]>0)) ? (sfn+1) % 10 : sfn;
       if(index==0){                       // get preamble
          pa = (unsigned short) gbits(dw[w],0,7);
          current1 = dw[w]; // remember this data word for later
          continue; 
       }

       if(index==1){                       // get sfid
          sfid = (unsigned short) gbits(dw[w],19,21);
          current2 = dw[w]; // remember this data word for later
          continue;
       }
          
       // get data based on sub frame id
       switch (sfid){
          
          case 1:   // get IODC if collecting ephemeris data
                    if( COLLECT_EPH && (index==7) )
                       xeph(dw[w], prn, index);
                    break;
          case 2:   // get ephemeris data
          case 3:   if(COLLECT_EPH)
                       xeph(dw[w], prn, index);
                    break;
          case 4:   // get almanac and utc data
                    if(COLLECT_UTC && (utcstructptr.utcmem == FALSE))
                       xutc(dw[w], prn, index);
          case 5:   // get almanac data
                    if(COLLECT_ALM)
                       xalm(dw[w], index);
                    break;
          default:  // not valid index
                    break; 
       }
     }
   }  
}  


/////////////////////////////////////////////////////////////////////////////////////////////////
// extract ephemeris data
//
void xeph(unsigned long dw, unsigned short prn, unsigned short index)
{
   long os=0;                           // offset from beginning of file 
   int count = 0;
   unsigned long     epharray[26];      // array for a local copy of ephemeris data for this sv

   // set availability flag for possibly new sv
   ephavail |= (unsigned long)pow(2,prn); // set the prn-th bit
   
   // read in data for this sv into local array for update
   os = prn * 26 * sizeof (unsigned long);                          // calculate file position offset
   rewind(stream6);
   fseek( stream6, os, SEEK_SET );               // find location
   count = fread( epharray, sizeof (unsigned long),26, stream6 );
   if(_Debug) showmessage( "Number items read from (eph.tmp):", 2, count );
   
   // extract emphemeris data
   showmessage("Extracting Ephemeris Data ..........",1,0);
   epharray[0] = pa;                                // pa previously extracted
   switch (index) {
      
      case 2:   // get sub frame 2 data
                if (sfid == 2){
                   epharray[2] = gbits(dw,0,7);     // IODE1
                   epharray[9] = gbits(dw,8,23);    // Crs
                }
                if (sfid == 3){
                   epharray[12] = gbits(dw,0,15);   // Cic
                   epharray[20] = gbits(dw,16,23);  // Omega01
                }
                break; 
      case 3:   // get sub frame 3 data
                if (sfid == 2){
                   epharray[5] = gbits(dw,0,15);    // deltan
                   epharray[14] = gbits(dw,16,23);  // M01
                }
                if (sfid == 3){
                   epharray[21] = gbits(dw,0,23);   // Omega02
                }
                break;
      case 4:   // get sub frame 4 data
                if (sfid == 2){
                   epharray[15] = gbits(dw,0,23);   // M02
                }  
                if (sfid == 3){
                   epharray[13] = gbits(dw,0,15);   // Cis
                   epharray[22] = gbits(dw,16,23);  // i01
                }
                break;         
      case 5:   // get sub frame 5 data
                if (sfid == 2){
                   epharray[10] = gbits(dw,0,15);   // Cuc
                   epharray[16] = gbits(dw,16,23);  // e1
                }
                if (sfid == 3){
                   epharray[23] = gbits(dw,0,23);   // i02
                }  
                break;
      case 6:   // get sub frame 6 data 
                if (sfid == 2){
                   epharray[17] = gbits(dw,0,23);   // e2
                }
                if (sfid == 3){
                   
                   epharray[8] = gbits(dw,0,15);    // Crc
                   epharray[24] = gbits(dw,16,23);  // omega1
                }
                break;
      case 7:   // get sub frame 7 data
                if (sfid == 1){
                   epharray[1] = gbits(dw,0,7);     // IODC
                }
                if (sfid == 2){
                   epharray[11] = gbits(dw,0,15);   // Cus
                   epharray[18] = gbits(dw,16,23);  // sqrA1
                }
                if (sfid == 3){
                   epharray[25] = gbits(dw,0,23);   // omega2
                }
                break;
      case 8:   // get sub frame 8 data
                if (sfid == 2){
                   epharray[19] = gbits(dw,0,23);   // sqrA2
                }
                if (sfid == 3){
                   epharray[7] = gbits(dw,0,23);    // OmegaDot
                }
                break;
      case 9:   // get sub frame 9 data 
                if (sfid == 2){
                   epharray[6] = gbits(dw,0,15);    // toe
                }
                if (sfid == 3){
                   epharray[3] = gbits(dw,0,7);     // IODE2
                   epharray[4] = gbits(dw,8,21);    // IDOT
                }
                break;
      default:  break;
   }
   
   // write data for this sv back to file
   rewind(stream6);
   fseek( stream6, os, SEEK_SET );               // find location
   count = fwrite( epharray, sizeof (unsigned long),26, stream6 );
   if(_Debug) showmessage( "Number items wrote to (eph.tmp):", 2, count );
}  // end xeph



/////////////////////////////////////////////////////////////////////////////////////////////////
// extract/process almanac data
//
void xalm(unsigned long dw, unsigned short index)
{
   long os=0;                           // offset from beginning of file 
   int count = 0;
   unsigned long     almarray[11],      // array for a local copy of almanac data for this sv
                     rawalmarray[11];   // array for a local copy of raw almanac data words for this sv
  
   if(index==2)
      svid = ( (unsigned short)gbits(dw,2,7) & 0x3f ) - 1;  //offset -1 for array index

   if(_Debug) showmessage( "Got svid = %hu", 2, svid );

   
   if( ( svid < 0 ) || ( svid > 31 ) ){
     if(_Debug) showmessage("Returning without action ..........",1,0);
     return;          // return on invalid alm svids(not between 0..31)
   }
   
   // set availability flag for possibly new sv
   almavail |= (unsigned long) pow(2,svid); // set the prn-th bit
   
   // read in data for this sv into local array for update
   os = svid * 11 * sizeof (unsigned long);                          // calculate file position offset
   rewind(stream7);
   rewind(stream8);   // raw data stream
   fseek( stream7, os, SEEK_SET );               // find location
   fseek( stream8, os, SEEK_SET );               // find location
   count = fread( almarray, sizeof (unsigned long),11, stream7 );
   if(_Debug) showmessage( "Number items read from (alm.tmp):", 2, count );
   count = fread( rawalmarray, sizeof (unsigned long),11, stream8 );
   if(_Debug) showmessage( "Number items read from (almanac.dat):", 2, count );
   
   // extract almanac data 
   showmessage("Extracting Almanac Data ..........",1,0);
   almarray[0] = pa;                                // pa previously extracted
   rawalmarray[1] = current1;                       // data word 1 previously found
   rawalmarray[2] = current2;                       // data word 2 previously found
   
   switch (index) {
      
      case 2:   // get sub frame 3 data
                almarray[1] = gbits(dw,8,23);     // e
                rawalmarray[3] = dw;              // data word 3 
                break;
                
      case 3:   // get sub frame 3 data
                almarray[2] = gbits(dw,0,7);      // toa
                almarray[3] = gbits(dw,8,23);     // deltai
                rawalmarray[4] = dw;              // data word 4 
                break;
      
      case 4:   // get sub frame 4 data
                almarray[4] = gbits(dw,0,15);     // OmegaDot
                rawalmarray[5] = dw;              // data word 5 
                break;         
      
      case 5:   // get sub frame 5 data
                almarray[5] = gbits(dw,0,23);     // sqrA
                rawalmarray[6] = dw;              // data word 6 
                break;
      
      case 6:   // get sub frame 6 data 
                almarray[6] = gbits(dw,0,23);     // Omega0
                rawalmarray[7] = dw;              // data word 7 
                break;
      
      case 7:   // get sub frame 7 data
                almarray[7] = gbits(dw,0,23);     // omega
                rawalmarray[8] = dw;              // data word 8 
                break;
      
      case 8:   // get sub frame 8 data
                almarray[8] = gbits(dw,0,23);     // M0
                rawalmarray[9] = dw;              // data word 9 
                break;
      
      case 9:   // get sub frame 9 data
                // get af0 parts and make 11bit data
                almarray[9] = gbits(dw,0,7);      // get 8 MSBs of af0
                almarray[9] <<= 3;                // shift for 3LSBs
                almarray[9] |= gbits(dw,19,21);   // or with 3LSBs
      
                almarray[10] = gbits(dw,8,18);     // af1
                rawalmarray[10] = dw;              // data word 10 
                break;
      
      default:  break;
   }
   
   // write data for this sv back to file
   rewind(stream7);
   rewind(stream8);
   fseek( stream7, os, SEEK_SET );               // find location
   fseek( stream8, os, SEEK_SET );               // find location
   count = fwrite( almarray, sizeof (unsigned long),11, stream7 );
   if(_Debug) showmessage( "Number items wrote to (alm.tmp):", 2, count );
   count = fwrite( rawalmarray, sizeof (unsigned long),11, stream8 );
   if(_Debug) showmessage( "Number items wrote to (almanac.dat):", 2, count );
}  // end xalm



/////////////////////////////////////////////////////////////////////////////////////////////////
// extract/process utc data
//
void xutc(unsigned long dw, unsigned short prn, unsigned short index)
{
   
   if(index==2){
      svid = (unsigned short) (gbits(dw,2,7) - 1);  //offset -1 for array index
      if( (utcstructptr.utcptr == NULL)&&(utcstructptr.utcmem == FALSE) ){ //need new data structure
         utcstructptr.utcptr = (utctype *)malloc( sizeof (utctype) );
         if( utcstructptr.utcptr == NULL ){
            utcstructptr.utcmem = TRUE;      // set out of memory flag
            cputs("\a");
            showmessage("ERROR - OUT OF MEMORY - UTC data will be lost",1,0);
            return;
         }
         else
            // initialize new structure
            _fmemset(utcstructptr.utcptr,0,sizeof(utctype));
      } 
   }
   if (utcstructptr.utcptr == NULL) return; 
   if( !(svid==55) )  return;   // return if not sv# 56 (remember svid was offset by -1)
   
   showmessage("Extracting UTC Data..........",1,0);
   switch (index) {
      case 5:   // get sub frame 5 data
                utcstructptr.utcptr->a1 = gbits(dw,0,23);
                break;
      case 6:   // get sub frame 6 data 
                utcstructptr.utcptr->a0 = gbits(dw,0,23); // 24 MSBs of a0
                break;
      case 7:   // get sub frame 7 data
                utcstructptr.utcptr->a0s = (unsigned short) gbits(dw,0,7);     // 8 LSBs of a0
                utcstructptr.utcptr->tot = gbits(dw,8,15);
                utcstructptr.utcptr->wnt = gbits(dw,16,23);
                break;
      case 8:   // get sub frame 8 data
                utcstructptr.utcptr->deltatls = gbits(dw,0,7);
                utcstructptr.utcptr->wnlsf = gbits(dw,8,15);
                utcstructptr.utcptr->dn = gbits(dw,16,23);
                break;
      case 9:   // get sub frame 9 data 
                utcstructptr.utcptr->deltatlsf = gbits(dw,0,7);
                break;
      default:  break;
   }
}  // end xutc

 
/////////////////////////////////////////////////////////////////////////////////////////////////
// get bits from data word
//
unsigned long gbits (unsigned long temp1, int first, int last)
{
   unsigned long tempbits=0;
   unsigned short tempbit=0,k,os=8; 

   for (k=first+os; k<=last+os; k++){
     tempbit = (unsigned short) ((temp1 >> (31-k)) & 0x01);
     tempbits = (tempbits << 1)|tempbit; 
   }
         
   return tempbits;
}  // end gbits

   
/////////////////////////////////////////////////////////////////////////////////////////////////
// make 32bit word from 8bit(MSB) and 24bit
//
unsigned long mword (unsigned long temp1, unsigned long temp2)
{
   
   temp1 <<= 24;
   temp1 |= temp2;
      
   return temp1;
}  // end mword


/////////////////////////////////////////////////////////////////////////////////////////////////
// converts an unsigned long interger to a scaled double
//
float UL2F(unsigned long temp1, int numbits, int scale, char type)
{
   varitype     invar;
   int          os = 32-numbits,Sign = 1;
   
   invar.ulval = temp1;
   
   // if scaled then determine if negative
   if ( (gbits(temp1,os-8,os-8)) & (type=='s') ) {
     Sign = -1;
     //convert to 1's complement - converting here directly to 2's complement causes
     //the overflow warning. 
     invar.ulval = 0xffffffff - invar.ulval;
     //convert to 2's complement
     invar.ulval++; 
   }  
     
   invar.fval = (float) (invar.ulval << os);

   return (invar.fval * Sign * (float)pow(2,scale - os));
}  // end ul2f 


/////////////////////////////////////////////////////////////////////////////////////////////////
// process/output requested sv data to output file (SVD.TXT)
//
void svout()
{
   unsigned short     prn;
   unsigned long      ulM0,ule,ulsqrA,ulOmega0,uli0,ulomega;
   float              IODC,IODE1,IODE2,IDOT,deltan,toe,OmegaDot,Crc,Crs,Cuc,Cus,Cic,Cis,M0,e,sqrA,
                      Omega0,i0,omega,toa,deltai,af0,af1,a0,a1,deltatls,tot,wnt,wnlsf,dn,deltatlsf,
                      PI= (float)3.14159265358979323846;
   long               os;   // offset
   int                count=0;                    
   
   ephconv     tempeph;      // union for converting eph structure and array
   almconv     tempalm;      // union for converting alm structure and array
   rawalmconv  temprawalm;   // union for converting alm structure and array
   
   tBOOL    match = FALSE;

   ///////////////////////////////////////////////
   // process ephemeris data for output to a file
   //
   if (COLLECT_EPH){
   for (prn=0; prn<32; prn++){

     // only check data collected unless in debug
     if( (!(ephavail & (unsigned long)pow(2,prn))) && (!_Debug) ) continue;   // check the prn-th bit and debug mode

     // read in data for this sv into local array for update
     os = prn * 26 * sizeof (unsigned long);                          // calculate file position offset
     rewind(stream6);
     fseek( stream6, os, SEEK_SET );               // find location
     count = fread( tempeph.epharray, sizeof (unsigned long),26, stream6 );
     if(_Debug) showmessage( "Number items read from (eph.tmp):", 2, count );
     
     // calculate a match of IODC/IODE1/IODE2 - zeros don't count
     match = ( (tempeph.ephstruct.IODC == tempeph.ephstruct.IODE1)
            && (tempeph.ephstruct.IODC == tempeph.ephstruct.IODE2)
            && (tempeph.ephstruct.IODC != 0) );
     
     // check preamble - if not correct then skip it - unless in debug mode
     if( ((tempeph.ephstruct.pa & 0x00ff) != 0x8b ) && (!_Debug) ){
        if(_Debug) fprintf(stream3,"\n\tEphemeris Data for SV %02hu has invalid preamble.  Data skipped\n",prn+1);
        continue;
     }
     
     // check for descrepencies with IODC, IODE1, and IODE2
     if ( !match ){
        showmessage("WARNING - IDOC/IDOE1/IDOE2 DON'T MATCH FOR SV#",2,prn+1);
        if(_Debug) fprintf(stream3,"\n\tEphemeris Data for SV %02hu has invalid data. IODC/IODE mismatch.\n",prn+1);
        continue;
     }
     
     // make conversions for ephemeris output
     ulM0     = mword(tempeph.ephstruct.M01,tempeph.ephstruct.M02);
     M0       = UL2F(ulM0,32,-31,'s') * PI;
   
     ule      = mword(tempeph.ephstruct.e1,tempeph.ephstruct.e2);
     e        = UL2F(ule,32,-33,'u');
                            
     ulsqrA   = mword(tempeph.ephstruct.sqrA1,tempeph.ephstruct.sqrA2);
     sqrA     = UL2F(ulsqrA,32,-19,'u');
   
     ulOmega0 = mword(tempeph.ephstruct.Omega01,tempeph.ephstruct.Omega02);
     Omega0   = UL2F(ulOmega0,32,-31,'s') * PI;
   
     uli0     = mword(tempeph.ephstruct.i01,tempeph.ephstruct.i02);
     i0       = UL2F(uli0,32,-31,'s') * PI;
   
     ulomega  = mword(tempeph.ephstruct.omega1,tempeph.ephstruct.omega2);
     omega    = UL2F(ulomega,32,-31,'s') * PI;
   
     IODC     = UL2F(tempeph.ephstruct.IODC,8,0,'u');
     IODE1    = UL2F(tempeph.ephstruct.IODE1,8,0,'u');
     IODE2    = UL2F(tempeph.ephstruct.IODE2,8,0,'u');
     Crs      = UL2F(tempeph.ephstruct.Crs,16,-5,'s');
     deltan   = UL2F(tempeph.ephstruct.deltan,16,-43,'s') * PI;
     Cuc      = UL2F(tempeph.ephstruct.Cuc,16,-29,'s');
     Cus      = UL2F(tempeph.ephstruct.Cus,16,-29,'s');
     toe      = UL2F(tempeph.ephstruct.toe,16,4,'u');
     Cic      = UL2F(tempeph.ephstruct.Cic,16,-29,'s');
     Cis      = UL2F(tempeph.ephstruct.Cis,16,-29,'s');
     Crc      = UL2F(tempeph.ephstruct.Crc,16,-5,'s');
     OmegaDot = UL2F(tempeph.ephstruct.OmegaDot,24,-43,'s') * PI;
     IDOT     = UL2F(tempeph.ephstruct.IDOT,14,-43,'s') * PI;
     // end conversions

     fprintf(stream3,"\n\t               Ephemeris Data for SV %02hu",prn+1);
     fprintf(stream3,"\n\t--------------------------------------------------------------\n");

     fprintf(stream3, "%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n",
             MA,M0,RD,MMD,deltan,RD,ECC,e,NU,SRA,sqrA,MT,LOPAN,Omega0,RD,IA,i0,RD,AP,omega,RD,RRA,OmegaDot,RD);
     fprintf(stream3, "%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n",
             RIA,IDOT,RD,LCC,Cuc,RD,LSC,Cus,RD,ORCC,Crc,MT,ORSC,Crs,MT,IACC,Cic,RD,IASC,Cis,RD,ERT,toe,SC);        
     fprintf(stream3, "%s%G%s\n%s%G%s\n%s%G%s\n\n",
             CID,IODC,NU,EID1,IODE1,NU,EID2,IODE2,NU);

     // hex dump test part only in _Debug Mode *********************************************
     if (_Debug){
       fprintf(stream3,"\n\t             _Debug Mode ******* PA is %02hx",tempeph.ephstruct.pa);
       fprintf(stream3,"\n\t             Ephemeris Hex Dump for SV %02hu",prn+1);
       fprintf(stream3,"\n\t--------------------------------------------------------------\n");
       fprintf(stream3, "%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n",
               MA,ulM0,MMD,tempeph.ephstruct.deltan,ECC,ule,SRA,ulsqrA,LOPAN,ulOmega0,
               IA,uli0,AP,ulomega,RRA,tempeph.ephstruct.OmegaDot);
       fprintf(stream3, "%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n%s%lx\n",
               RIA,tempeph.ephstruct.IDOT,LCC,tempeph.ephstruct.Cuc,LSC,tempeph.ephstruct.Cus,
               ORCC,tempeph.ephstruct.Crc,ORSC,tempeph.ephstruct.Crs,IACC,tempeph.ephstruct.Cic,
               IASC,tempeph.ephstruct.Cis,ERT,tempeph.ephstruct.toe);        
       fprintf(stream3, "%s%lx\n%s%lx\n%s%lx\n\n",
               CID,tempeph.ephstruct.IODC,EID1,tempeph.ephstruct.IODE1,EID2,tempeph.ephstruct.IODE2);
     }
     // end hex dump test *******************************************************************     
   } 
   } // end COLLECT_EPH
   
   /////////////////////////////////////////////
   // process almanac data for output to a file
   //
   if(COLLECT_ALM){
   for (prn=0; prn<32; prn++){

     // only check data collected unless in debug
     if( (!(almavail & (unsigned long)pow(2,prn))) && (!_Debug) ) continue;   // check the prn-th bit and debug mode

     // read in data for this sv into local array for update
     os = prn * 11 * sizeof (unsigned long);                          // calculate file position offset
     rewind(stream7);
     fseek( stream7, os, SEEK_SET );               // find location
     count = fread( tempalm.almarray, sizeof (unsigned long),11, stream7 );
     if(_Debug) showmessage( "Number items read from (alm.tmp):", 2, count );
   
     // check preamble - if not correct then skip it - unless in debug mode
     if( ((tempalm.almstruct.pa &= 0x00ff) != 0x8b ) && (!_Debug) ){
        if(_Debug) fprintf(stream3,"\n\tAlmanac Data for SV %02hu has invalid preamble.  Data skipped\n",prn+1);
        continue;
     }
     
     // make conversions for almanac output
     e         = UL2F(tempalm.almstruct.e,16,-21,'u');
     toa       = UL2F(tempalm.almstruct.toa,8,12,'u');
     deltai    = UL2F(tempalm.almstruct.deltai,16,-19,'s') * PI;
     OmegaDot  = UL2F(tempalm.almstruct.OmegaDot,16,-38,'s') * PI;
     sqrA      = UL2F(tempalm.almstruct.sqrA,24,-11,'u');
     Omega0    = UL2F(tempalm.almstruct.Omega0,24,-23,'s') * PI;
     omega     = UL2F(tempalm.almstruct.omega,24,-23,'s') * PI;
     M0        = UL2F(tempalm.almstruct.M0,24,-23,'s') * PI;
     af0       = UL2F(tempalm.almstruct.af0,11,-20,'s');
     af1       = UL2F(tempalm.almstruct.af1,11,-38,'s');
     // end conversions

     fprintf(stream3,"\n\t                Almanac Data for SV %02hu",prn+1);
     fprintf(stream3,"\n\t--------------------------------------------------------------\n");

     fprintf(stream3, "%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n",
             ECC,e,NU,ART,toa,SC,DELI,deltai,RD,RRA,OmegaDot,RD,SRA,sqrA,MT,LOPAN,Omega0,RD,AP,omega,RD,
             MA,M0,RD,AF0,af0,SC,AF1,af1,SPS);

     // hex dump test part only in _Debug Mode **********************************************
     if (_Debug){
       fprintf(stream3,"\n\t             _Debug Mode ******* PA is %02hx",tempalm.almstruct.pa);
       fprintf(stream3,"\n\t             Almanac Hex Dump for SV %02hu",prn+1);
       fprintf(stream3,"\n\t--------------------------------------------------------------\n");
       fprintf(stream3, "%s% lx\n%s% lx\n%s% lx\n%s% lx\n%s% lx\n%s% lx\n%s% lx\n%s% lx\n%s% lx\n%s% lx\n",
               ECC,tempalm.almstruct.e,ART,tempalm.almstruct.toa,DELI,tempalm.almstruct.deltai,
               RRA,tempalm.almstruct.OmegaDot,SRA,tempalm.almstruct.sqrA,LOPAN,tempalm.almstruct.Omega0,
               AP,tempalm.almstruct.omega,MA,tempalm.almstruct.M0,AF0,tempalm.almstruct.af0,
               AF1,tempalm.almstruct.af1);
     }
     // end hex dump test *******************************************************************     
   }
   
   
   /////////////////////////////////////////////
   // process almanac raw data for output to a file
   //

   fprintf(almdata,"\n\t                      Raw Almanac Data");
   fprintf(almdata,"\n\t--------------------------------------------------------------\n\n");
   
   for (prn=0; prn<32; prn++){

     // only check data collected unless in debug
     if( (!(almavail & (unsigned long)pow(2,prn))) && (!_Debug) ) continue;   // check the prn-th bit and debug mode

     // read in data for this sv into local array for update
     os = prn * 11 * sizeof (unsigned long);                          // calculate file position offset
     rewind(stream8);
     fseek( stream8, os, SEEK_SET );               // find location
     count = fread( temprawalm.rawalmarray, sizeof (unsigned long),11, stream8 );
     if(_Debug) showmessage( "Number items read from (rawalm.tmp):", 2, count );
   
     // check preamble - if not correct then skip it - unless in debug mode
     if( ((tempalm.almstruct.pa &= 0x00ff) != 0x8b ) && (!_Debug) ){
        if(_Debug) fprintf(stream3,"\n\tAlmanac Data for SV %02hu has invalid preamble.  Data skipped\n",prn+1);
        continue;
     }
     
     fprintf(almdata, "SV# %02hu\t%08lx\t%08lx\t%08lx\t%08lx\n      \t%08lx\t%08lx\t%08lx\t%08lx\n\n",
             prn+1,temprawalm.rawalmstruct.dw3,temprawalm.rawalmstruct.dw4,temprawalm.rawalmstruct.dw5,
             temprawalm.rawalmstruct.dw6,temprawalm.rawalmstruct.dw7,temprawalm.rawalmstruct.dw8,
             temprawalm.rawalmstruct.dw9,temprawalm.rawalmstruct.dw10);
   }
   } // end COLLECT_ALM

   /////////////////////////////////////////
   // process utc data for output to a file
   //
   if(COLLECT_UTC){
      if( utcstructptr.utcptr == NULL ) return;   // if no data then don't do anything
      // build a0 from its parts
      (utcstructptr.utcptr)->a0 <<= 8;               // make room for the 8 LSBs
      (utcstructptr.utcptr)->a0 |= (utcstructptr.utcptr)->a0s;    // merge LSBs to end
      // make conversions for utc output
      a0        = UL2F((utcstructptr.utcptr)->a0,32,-30,'s');
      a1        = UL2F((utcstructptr.utcptr)->a1,24,-50,'s');
      deltatls  = UL2F((utcstructptr.utcptr)->deltatls,8,0,'s');
      tot       = UL2F((utcstructptr.utcptr)->tot,8,12,'u');
      wnt       = UL2F((utcstructptr.utcptr)->wnt,8,0,'u');
      wnlsf     = UL2F((utcstructptr.utcptr)->wnlsf,8,0,'u');
      dn        = UL2F((utcstructptr.utcptr)->dn,8,0,'u');
      deltatlsf = UL2F((utcstructptr.utcptr)->deltatlsf,8,0,'s');
      // end conversions

     fprintf(stream3,"\n\t                          UTC Data");
     fprintf(stream3,"\n\t--------------------------------------------------------------\n");

     fprintf(stream3, "%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n%s% G%s\n",
             A0,a0,SC,A1,a1,SPS,DTLS,deltatls,SC,TOT,tot,SC,WNT,wnt,WK,WNLSF,wnlsf,WK,DN,dn,DY,
             DTLSF,deltatlsf,SC);

     // hex dump test part only in _Debug Mode *********************************************
     if (_Debug){
//       fprintf(stream3,"\n\t             _Debug Mode ******* PA is %02hx",tempeph.pa);
       fprintf(stream3,"\n\t             UTC Hex Dump" );
       fprintf(stream3,"\n\t--------------------------------------------------------------\n");
       fprintf(stream3, "%s% lx%s\n%s% lx%s\n%s% lx%s\n%s% lx%s\n%s% lx%s\n%s% lx%s\n%s% lx%s\n%s% lx%s\n",
               A0,(utcstructptr.utcptr)->a0,SC,A1,(utcstructptr.utcptr)->a1,SPS,DTLS,(utcstructptr.utcptr)->deltatls,SC,TOT,(utcstructptr.utcptr)->tot,SC,
               WNT,(utcstructptr.utcptr)->wnt,WK,WNLSF,(utcstructptr.utcptr)->wnlsf,WK,DN,(utcstructptr.utcptr)->dn,DY,DTLSF,(utcstructptr.utcptr)->deltatlsf,SC);
     }
     // end hex dump test *******************************************************************     
   } // end COLLECT_UTC
   
   if( COLLECT_EPH ){
      fclose(stream6);
      if( !_Debug ) remove("eph.tmp"); // don't remove if in debug mode
   }
   if( COLLECT_ALM ){
      fclose(stream7);
      fclose(stream8);
      fclose (almdata);                // close raw almanac output file
      if( !_Debug ) remove("alm.tmp"); // don't remove if in debug mode
   } 
 
}

/////////////////////////////////////////////////////////////////////////////////////////////////
// display a message to the labmon message line
//
void showmessage(char message[], unsigned short type, unsigned short number)
{
   char msgbuff[80];                 // buffer for output message

   clear_message_line();
   if( type == 1 )
      sprintf(msgbuff,"%s",message);
   if( type == 2 )
      sprintf(msgbuff,"%s %d",message,number);
   
   _outtext(msgbuff);
}

       